    import { spawn } from 'child_process';
    import path from 'path';
    import createError from 'http-errors';
    import RawFile from '../models/rawFile.js';

    class PersianProcessorController {
        /**
         * Processes Persian text in a batch of files
         */
        static async processBatch(req, res, next) {
            try {
                const { batchId } = req.params;

                // Verify batch exists and has Persian content
                const files = await RawFile.find({ batchId });
                if (!files.length) {
                    throw createError(404, `No files found for batch ${batchId}`);
                }

                const hasPersianContent = files.some(file => 
                    file.metadata?.languages?.includes('fa')
                );

                if (!hasPersianContent) {
                    throw createError(400, 'No Persian content found in batch');
                }

                // Run Python processor
                const pythonScript = path.join(process.cwd(), 'src', 'AI-PipeLine', 'utils', 'persian_processor.py');
                const pythonProcess = spawn('python', [pythonScript, batchId]);

                let output = '';
                let error = '';

                pythonProcess.stdout.on('data', (data) => {
                    output += data.toString();
                });

                pythonProcess.stderr.on('data', (data) => {
                    error += data.toString();
                });

                await new Promise((resolve, reject) => {
                    pythonProcess.on('close', (code) => {
                        if (code !== 0) {
                            reject(new Error(`Persian processing failed: ${error}`));
                        } else {
                            resolve();
                        }
                    });
                });

                // Parse and return results
                const results = JSON.parse(output);
                res.json({
                    message: 'Persian text processing complete',
                    batchId,
                    results
                });

            } catch (error) {
                next(createError(500, `Persian processing error: ${error.message}`));
            }
        }

        /**
         * Gets the processing status and results for a batch
         */
        static async getBatchResults(req, res, next) {
            try {
                const { batchId } = req.params;

                const results = await RawFile.collection
                    .aggregate([
                        { $match: { batchId } },
                        { $lookup: {
                            from: 'processed_persian',
                            localField: 'batchId',
                            foreignField: 'batchId',
                            as: 'persianProcessing'
                        }},
                        { $project: {
                            _id: 0,
                            fileId: '$_id',
                            originalName: 1,
                            metadata: 1,
                            processingHistory: {
                                $filter: {
                                    input: '$processingHistory',
                                    as: 'history',
                                    cond: { $eq: ['$$history.stage', 'persian_nlp'] }
                                }
                            },
                            persianResults: { $arrayElemAt: ['$persianProcessing', 0] }
                        }}
                    ]).toArray();

                if (!results.length) {
                    throw createError(404, `No results found for batch ${batchId}`);
                }

                res.json({
                    batchId,
                    files: results
                });

            } catch (error) {
                next(error);
            }
        }

        /**
         * Gets consolidated Persian NLP analytics for a batch
         */
        static async getBatchAnalytics(req, res, next) {
            try {
                const { batchId } = req.params;

                const analytics = await RawFile.collection
                    .aggregate([
                        { $match: { batchId } },
                        { $lookup: {
                            from: 'processed_persian',
                            localField: 'batchId',
                            foreignField: 'batchId',
                            as: 'persianProcessing'
                        }},
                        { $unwind: '$persianProcessing' },
                        { $unwind: '$persianProcessing.processed_files' },
                        { $group: {
                            _id: '$batchId',
                            totalFiles: { $sum: 1 },
                            uniqueLemmas: { $addToSet: '$persianProcessing.processed_files.processed_content.lemmas' },
                            commonPhrases: { $push: '$persianProcessing.processed_files.processed_content.key_phrases' },
                            processingTimes: { 
                                $push: {
                                    fileId: '$persianProcessing.processed_files.fileId',
                                    processedAt: '$persianProcessing.processed_files.metadata.processedAt'
                                }
                            }
                        }},
                        { $project: {
                            _id: 0,
                            batchId: '$_id',
                            totalFiles: 1,
                            uniqueLemmasCount: { $size: '$uniqueLemmas' },
                            commonPhrases: {
                                $reduce: {
                                    input: '$commonPhrases',
                                    initialValue: [],
                                    in: { $setUnion: ['$$value', '$$this'] }
                                }
                            },
                            processingTimes: 1
                        }}
                    ]).toArray();

                if (!analytics.length) {
                    throw createError(404, `No analytics found for batch ${batchId}`);
                }

                res.json(analytics[0]);

            } catch (error) {
                next(error);
            }
        }
    }

    export default PersianProcessorController;