import fs from 'fs/promises';
import path from 'path';
import { PythonProcessManager } from './PythonProcessManager.js';
import { TimeEstimator } from './TimeEstimator.js';
import { spawn } from 'child_process';
import { fileURLToPath } from 'url';

export class FileProcessor {
    constructor(agentFileService) {
        this.agentFileService = agentFileService;
        const __filename = fileURLToPath(import.meta.url);
        const __dirname = path.dirname(__filename);
        this.pythonScriptPath = path.join(__dirname, '../../../../python/document_processor.py');
        this.pythonManager = new PythonProcessManager();
        this.timeEstimator = new TimeEstimator();
    }

    /**
     * Process and upload files to OpenAI
     */
    async processAndUploadFiles(agentId, outputDir) {
        try {
            const resultsPath = path.join(outputDir, 'processing_results.json');
            
            try {
                // Check if results file exists
                await fs.access(resultsPath);
                
                // Upload the processing results file directly
                const uploadResult = await this.agentFileService.upload_file(agentId, resultsPath);
                console.log(`Successfully uploaded processing results file`);
                
                return [uploadResult];
            } catch (error) {
                console.error('Failed to upload processing results:', error);
                throw new Error('Processing results file not found or upload failed');
            }
        } catch (error) {
            console.error('Error in processAndUploadFiles:', error);
            throw error;
        }
    }

    /**
     * Calculate estimated completion time
     */
    async calculateEstimatedTime(inputDir) {
        try {
            await fs.access(inputDir);
            return this.timeEstimator.calculateEstimatedTime(inputDir);
        } catch (error) {
            // If directory doesn't exist, create it and return minimal time estimate
            await fs.mkdir(inputDir, { recursive: true });
            return new Date(Date.now() + 60000); // 1 minute from now
        }
    }

    /**
     * Process documents through Python script
     */
    async processDocuments(inputDir, outputDir, onProgress) {
        // Ensure both directories exist
        await fs.mkdir(inputDir, { recursive: true });
        await fs.mkdir(outputDir, { recursive: true });
        
        // Check if there are any files to process
        const files = await fs.readdir(inputDir);
        if (files.length === 0) {
            // No files to process, return early
            if (onProgress) {
                await onProgress(100);
            }
            return;
        }
        
        return this.pythonManager.runProcessor(inputDir, outputDir, onProgress);
    }
}