import { spawn } from 'child_process';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

export class PythonProcessManager {
    /**
     * Run Python document processor script
     */
    async runProcessor(inputDir, outputDir, onProgress) {
        try {
            const pythonScript = path.join(__dirname, '..', '..', 'utils', 'document_processor.py');
            
            // Convert paths to absolute
            const absoluteInputDir = path.resolve(inputDir);
            const absoluteOutputDir = path.resolve(outputDir);

            // Use 'python3' if available, fall back to 'python'
            let pythonCommand = 'python3';
            try {
                await new Promise((resolve, reject) => {
                    const testProcess = spawn(pythonCommand, ['--version']);
                    testProcess.on('error', () => {
                        pythonCommand = 'python';
                        resolve();
                    });
                    testProcess.on('close', (code) => {
                        if (code === 0) resolve();
                        else {
                            pythonCommand = 'python';
                            resolve();
                        }
                    });
                });
            } catch (error) {
                pythonCommand = 'python';
            }

            const pythonProcess = spawn(pythonCommand, [
                pythonScript,
                absoluteInputDir,
                absoluteOutputDir
            ]);

            let output = '';
            let error = '';
            const progressPattern = /Progress: (\d+)%/;
            const infoPattern = /\[\s*INFO\s*\]/i;

            pythonProcess.stdout.on('data', (data) => {
                const text = data.toString();
                output += text;
                
                // Log stdout messages
                console.log(`Python Process Output: ${text.trim()}`);
                
                // Update progress if found in output
                const match = text.match(progressPattern);
                if (match && onProgress) {
                    const progress = parseInt(match[1]);
                    onProgress(progress);
                }
            });

            pythonProcess.stderr.on('data', (data) => {
                const text = data.toString();
                // Check if it's an info message
                if (infoPattern.test(text)) {
                    console.log(`Python Info: ${text.trim()}`);
                } else {
                    error += text;
                    console.error(`Python Error: ${text.trim()}`);
                }
            });

            return new Promise((resolve, reject) => {
                pythonProcess.on('close', (code) => {
                    if (code !== 0 && error) {
                        console.error('Process failed with code:', code);
                        console.error('Error details:', error);
                        reject(new Error(`Processing failed with code ${code}: ${error}`));
                        return;
                    }
                    resolve({ output, code });
                });

                pythonProcess.on('error', (err) => {
                    console.error('Process spawn error:', err);
                    reject(new Error(`Failed to start Python process: ${err.message}`));
                });
            });
        } catch (error) {
            console.error('Python process execution error:', error);
            throw new Error(`Python process execution failed: ${error.message}`);
        }
    }
}