import { AgentQueueModel } from "../../models/agent_queues.js";
import mongoose from "mongoose";

export class QueueManager {
  async withOptionalTransaction(callback) {
    let session;
    try {
      // Try to start a session
      session = await mongoose.startSession();
      
      // Check if transactions are supported
      if (session.supportTransaction) {
        try {
          const result = await session.withTransaction(callback);
          return result;
        } catch (error) {
          throw error;
        }
      } else {
        // If transactions aren't supported, run without transaction
        const result = await callback(session);
        return result;
      }
    } catch (error) {
      throw error;
    } finally {
      if (session) {
        session.endSession();
      }
    }
  }

  /**
   * Update queue positions for all pending items
   */
  async updateQueuePositions() {
    await this.withOptionalTransaction(async (session) => {
      const queryOptions = session ? { session } : {};
      const activeItems = await AgentQueueModel.find({
        status: { $in: ["queued", "processing"] }
      }).sort("queue_position").session(session);
      
      // Batch updates for better performance
      const bulkOps = [];

      for (let i = 0; i < activeItems.length; i++) {
        if (activeItems[i].queue_position !== i + 1) {
          bulkOps.push({
            updateOne: {
              filter: { _id: activeItems[i]._id },
              update: { $set: { queue_position: i + 1 } }
            }
          });
        }
      }
      if (bulkOps.length > 0) {
        await AgentQueueModel.bulkWrite(bulkOps, queryOptions);
      }
    });
  }

  /**
   * Create new queue entry
   * @param {string} agentId - The ID of the agent
   * @param {string} exhibitionId - The ID of the exhibition
   * @param {string} boughtenBoothId - The ID of the boughten booth
   * @param {Date} estimatedCompletionTime - Estimated completion time
   * @returns {Promise<Document>} The created queue entry
   */
  async createQueueEntry(agentId, exhibitionId, boughtenBoothId, estimatedCompletionTime) {
    return await this.withOptionalTransaction(async (session) => {
      const queryOptions = session ? { session } : {};
      const lastQueueItem = await AgentQueueModel.findOne({
        status: { $in: ["queued", "processing"] }    
      }).sort("-queue_position").session(session);

      const queuePosition = (lastQueueItem?.queue_position || 0) + 1;

      const queueStatus = new AgentQueueModel({
        agent_id: agentId,
        exhibition_id: exhibitionId,
        boughten_booth_id: boughtenBoothId,
        queue_position: queuePosition,
        estimated_completion_time: estimatedCompletionTime,
        status: "queued",
        processing_details: {
          total_files: 0,
          processed_files: 0,
          failed_files: 0
        }
      });
   
      return await queueStatus.save(queryOptions);
    });
  }

  /**
   * Get queue status for specific agent
   * @param {string} agentId - The ID of the agent to get queue status for
   */
  async getAgentQueue(agentId) {
    const status = await AgentQueueModel.findOne({
      agent_id: agentId,
      status: { $in: ["queued", "processing"] }
    });
    return status;
  }

  /**
   * Clean up completed or failed queue entries older than specified time
   * @param {number} maxAgeHours - Maximum age in hours before cleanup
   */
  async cleanupOldEntries(maxAgeHours = 1) {
    const cutoffDate = new Date(Date.now() - (maxAgeHours * 60 * 60 * 1000));
    await AgentQueueModel.deleteMany({
      status: { $in: ["completed", "failed"] },
      "processing_details.completion_time": { $lt: cutoffDate }
    });
  }

  /**
   * Check for and handle stalled queue items
   * @param {number} stallThresholdMinutes - Minutes before considering an item stalled
   * @returns {Promise<void>}
   */
  async handleStalledJobs(stallThresholdMinutes = 30) {
    await this.withOptionalTransaction(async (session) => {
      const queryOptions = session ? { session } : {};
      const stallThreshold = new Date(Date.now() - (stallThresholdMinutes * 60 * 1000));
      const stalledJobs = await AgentQueueModel.find({
        status: "processing",
        "processing_details.last_update_time": { $lt: stallThreshold }
      }).session(session);

      for (const job of stalledJobs) {
        // Reset stalled job back to queued state
        job.status = "queued";
        job.progress = 0;
        job.processing_details.processed_files = 0;
        job.processing_details.failed_files = 0;
        await job.save(queryOptions);
      }

      await this.updateQueuePositions();
    });
  }
}
