import { PipelineConfig } from './config/pipeline-config.js';
import { FileService } from './services/file-service.js';
import { OpenAIService } from './services/openai-service.js';
import { PipelineError } from './errors/pipeline-error.js';
import chatRoutes from './routes/chat.js';
import agentRoutes from './routes/agent.js';

/**
 * AI Pipeline for managing OpenAI assistants and file processing
 */
class AIPipeline {
    constructor() {
        this.initialized = false;
        this.fileService = null;
        this.openaiService = null;
    }

    /**
     * Initialize the AI Pipeline and attach it to an existing Express app
     * @param {Object} app Express application instance
     * @param {Object} config Configuration object
     * @returns {Promise<void>}
     */
    async initialize(app, config) {
        if (this.initialized) {
            throw PipelineError.initialization('AI Pipeline already initialized');
        }

        if (!app || typeof app.use !== 'function') {
            throw PipelineError.configuration('Valid Express app instance required');
        }

        try {
            // Validate configuration
            const validatedConfig = PipelineConfig.validate(config);

            // Initialize services
            this.fileService = new FileService(validatedConfig.uploadDir);
            await this.fileService.initialize();

            this.openaiService = new OpenAIService(validatedConfig.openaiApiKey);

            // Mount routes
            app.use('/agent/chat', chatRoutes);
            app.use('/agent', agentRoutes);

            // Add error handling middleware
            this.#setupErrorHandling(app, validatedConfig.environment);

            this.initialized = true;
            console.log(`AI Pipeline initialized with upload directory: ${validatedConfig.uploadDir}`);
        } catch (error) {
            const pipelineError = error instanceof PipelineError 
                ? error 
                : PipelineError.initialization('AI Pipeline initialization failed', error);
            console.error('Initialization error:', pipelineError);
            throw pipelineError;
        }
    }

    /**
     * Get the configured upload directory
     * @returns {string} Upload directory path
     */
    getUploadDir() {
        this.#ensureInitialized();
        return this.fileService.getUploadDir();
    }

    /**
     * Creates an OpenAI assistant
     * @param {Object} config Assistant configuration
     * @returns {Promise<string>} Assistant ID
     */
    async createAssistant(config) {
        this.#ensureInitialized();
        
        try {
            const validatedConfig = PipelineConfig.validateAssistantConfig(config);
            const assistant = await this.openaiService.createAssistant(validatedConfig);
            return assistant.id;
        } catch (error) {
            throw PipelineError.operation('Failed to create assistant', error);
        }
    }

    /**
     * Creates an assistant with file search capability
     * @param {Object} config Assistant configuration
     * @returns {Promise<Object>} Created assistant with vector store attached
     */
    async createAssistantWithFileSearch(config) {
        this.#ensureInitialized();

        try {
            const validatedConfig = PipelineConfig.validateAssistantConfig(config);
            return await this.openaiService.createAssistantWithFileSearch(validatedConfig);
        } catch (error) {
            throw PipelineError.operation('Failed to create assistant with file search', error);
        }
    }

    /**
     * Cleanup resources when shutting down
     * @returns {Promise<void>}
     */
    async cleanup() {
        if (!this.initialized) {
            return;
        }

        try {
            await this.fileService?.cleanup();
            this.initialized = false;
            console.log('AI Pipeline cleaned up successfully');
        } catch (error) {
            throw PipelineError.operation('Error during cleanup', error);
        }
    }

    /**
     * Setup error handling middleware
     * @private
     */
    #setupErrorHandling(app, environment) {
        app.use((err, req, res, next) => {
            if (res.headersSent) {
                return next(err);
            }

            const error = err instanceof PipelineError ? err : new PipelineError(err.message, 'UNKNOWN', err);

            console.error('AI Pipeline Error:', {
                code: error.code,
                message: error.message,
                stack: error.stack,
                cause: error.cause,
                timestamp: error.timestamp
            });

            res.status(500).json({
                error: environment === 'production' ? 'Internal Server Error' : error.message,
                code: error.code,
                status: 500,
                timestamp: error.timestamp,
                stack: environment === 'production' ? undefined : error.stack
            });
        });
    }

    /**
     * Ensure pipeline is initialized before operations
     * @private
     */
    #ensureInitialized() {
        if (!this.initialized) {
            throw PipelineError.operation('AI Pipeline not initialized');
        }
    }
}

// Create and export singleton instance
export default new AIPipeline();