import mongoose from "mongoose";

const AgentSchema = new mongoose.Schema(
  {
    ref: {
      type: String,
      required: true,
    },
    ref_id: {
      type: mongoose.Schema.Types.ObjectId,
      required: true,
    },
    name: {
      type: String,
      required: true,
    },
    brand_guidelines: {
      type: String,
      required: true,
    },
    available_topics: [
      {
        type: String,
        required: true,
      },
    ],
    raw_files: [
      {
        file_id: {
          type: mongoose.Schema.Types.ObjectId,
          required: true,
          ref: "files",
        },
      },
    ],
    processed_files: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "files",
    },
    queue_status: {
      type: String,
      enum: ["not_queued", "queued", "processing", "completed", "failed"],
      default: "not_queued",
    },
    system_prompt: {
      type: String,
      required: true,
    },
    openai_assistant_id: {
      type: String,
      required: false,
    },
    openai_vector_store_id: {
      type: String,
    },
    last_processed_at: {
      type: Date,
    },
    processing_attempts: {
      type: Number,
      default: 0,
    },
    is_removed: {
      type: Boolean,
      required: true,
      default: false,
    },
  },
  {
    timestamps: true,
  }
);

// Indexes for efficient lookups
AgentSchema.index({ ref: 1, ref_id: 1 });
AgentSchema.index({ queue_status: 1 });
AgentSchema.index({ is_removed: 1 });
AgentSchema.index({ "raw_files.processing_status": 1 });

// Instance method to append new files
AgentSchema.methods.append_raw_files = async function (files) {
  const file_entries = Array.isArray(files) ? files : [files];

  const formatted_entries = file_entries.map((file) => ({
    file_url: file.url,
    file_metadata: {
      original_name: file.original_name,
      size_bytes: file.size,
      mime_type: file.mime_type,
    },
  }));

  this.raw_files.push(...formatted_entries);
  return await this.save();
};

// Static method to find agent and append files
AgentSchema.statics.append_raw_files_to_agent = async function (
  agent_id,
  files
) {
  const agent = await this.findById(agent_id);
  if (!agent) {
    throw new Error("Agent not found");
  }

  return await agent.append_raw_files(files);
};

// Method to update file processing status
AgentSchema.methods.update_file_status = async function (
  file_url,
  status,
  metadata = {}
) {
  const file_entry = this.raw_files.find((f) => f.file_url === file_url);
  if (!file_entry) {
    throw new Error("File not found in agent records");
  }

  file_entry.processing_status = status;
  if (metadata.openai_file_id) {
    file_entry.openai_file_id = metadata.openai_file_id;
  }
  if (metadata.error) {
    file_entry.file_metadata.processing_error = metadata.error;
  }
  if (status === "processed" || status === "failed") {
    file_entry.file_metadata.processed_at = new Date();
  }

  return await this.save();
};

// Method to track processing attempts and errors
AgentSchema.methods.log_processing_attempt = async function (errorData) {
  this.processing_attempts += 1;
  this.last_processed_at = new Date();

  if (errorData) {
    this.error_logs.push({
      error_message: errorData.error,
      error_type: "ProcessingError",
      file_url: errorData.file_url,
      retry_count: errorData.retry_count || 0,
    });
  }

  return await this.save();
};

export const AgentModel = mongoose.model("agents", AgentSchema);
