import { Router } from "express";
import { AgentModel } from "../models/agent.js";
import { BoughtenBoothsModel } from "../../models/boughten_booths.js";
import { OpenAIService } from "../services/openai-service.js";
import { generateSystemPrompt } from "../services/prompt-service.js";
import { AgentFileService } from "../services/agent-file-service.js";
import mongoose from "mongoose";
import DocumentProcessingController from "../controllers/documentProcessing.js";
import { AgentSwaggerDocs } from "../docs/agent.swagger.js";
import { body, validationResult, param } from "express-validator";

const router = Router();
const openAIService = new OpenAIService(process.env.OPENAI_API_KEY);
const agentFileService = new AgentFileService(process.env.OPENAI_API_KEY);

// Add helper function at the top level
const sanitizeError = (error) => {
  // Log the full error for debugging
  console.error("Original error:", error);

  if (error.message?.includes("openai") || error.message?.includes("OpenAI")) {
    return "An error occurred while processing with AI services";
  }
  if (error.message?.includes("validation")) {
    return "Invalid input data provided";
  }
  if (error.message?.includes("duplicate")) {
    return "This resource already exists";
  }
  return "An unexpected error occurred";
};

/**
 * @swagger
 * tags:
 *   name: AI Pipeline
 *   description: AI agent management endpoints
 */

/**
 * @swagger
 * /agent/create:
 *   post:
 *     $ref: '#/paths/~1agent~1create/post'
 */
router.post(
  "/create",
  [
    body("name").notEmpty().withMessage("Name is required"),
    body("brand_guidelines")
      .notEmpty()
      .withMessage("Brand guidelines are required"),
    body("available_topics")
      .notEmpty()
      .withMessage("Available topics are required"),
    body("ref")
      .notEmpty()
      .withMessage("Reference is required")
      .isIn(["boughten_booths"])
      .withMessage("Invalid reference type"),
    body("ref_id")
      .notEmpty()
      .withMessage("Reference id is required")
      .isMongoId()
      .withMessage("Invalid Mongo ID for ref_id"),
  ],
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }
    try {
      const { name, brand_guidelines, available_topics, ref, ref_id } =
        req.body;

      // Validate required fields
      if (!name || !brand_guidelines || !available_topics || !ref || !ref_id) {
        return res.status(400).json({
          error: "Missing required fields",
        });
      }

      // Get the correct model based on the reference
      let RefModel;
      if (ref === "boughten_booths") {
        RefModel = BoughtenBoothsModel;
      } else {
        return res.status(400).json({
          error: "Unsupported reference type",
        });
      }

      const owner = await RefModel.findById(ref_id);

      if (!owner) {
        return res.status(404).json({
          error: "Referenced owner not found",
        });
      }

      // Make sure owner has creator_name field
      if (!owner.creator_name) {
        return res.status(400).json({
          error: "Owner record missing creator_name field",
        });
      }

      // Generate system prompt using creator_name from owner record
      const systemPrompt = await generateSystemPrompt(
        owner.creator_name,
        brand_guidelines,
        available_topics
      );

      // Create agent document immediately
      const agent = new AgentModel({
        name,
        brand_guidelines,
        available_topics,
        ref,
        ref_id,
        system_prompt: systemPrompt,
        queue_status: "processing", // Initial status while assistant is being created
        raw_files: [],
      });

      await agent.save();

      // Update the referenced model to include this agent's ID
      try {
        await RefModel.findByIdAndUpdate(ref_id, {
          $push: { agent_ids: agent._id },
        });
      } catch (error) {
        console.error("Error updating referenced model:", error);
        // Continue execution since agent was created successfully
      }

      // Return agent_id immediately
      res.status(201).json({
        agent_id: agent._id,
      });

      // Asynchronously create OpenAI assistant
      (async () => {
        try {
          const assistantConfig = {
            name: name,
            instructions: systemPrompt,
            model: "gpt-4o-mini",
          };
          console.log(
            "[DEBUG] Creating OpenAI assistant with config:",
            assistantConfig
          );

          const { assistant } =
            await openAIService.create_assistant_with_file_search(
              assistantConfig
            );
          console.log("[DEBUG] OpenAI service returned assistant:", assistant);

          if (!assistant || !assistant.id) {
            console.error(
              "[ERROR] Assistant creation did not return a valid id:",
              assistant
            );
            throw new Error("Assistant creation failed: Missing id");
          }

          // Update agent with assistant ID
          await AgentModel.findByIdAndUpdate(agent._id, {
            openai_assistant_id: assistant.id,
            queue_status: "not_queued", // Reset to default status after successful creation
          });
        } catch (error) {
          console.error("Error creating OpenAI assistant:", error);
          await AgentModel.findByIdAndUpdate(agent._id, {
            queue_status: "failed",
            error_logs: [
              {
                error_message: error.message,
                error_type: "AssistantCreationError",
              },
            ],
          });
        }
      })();
    } catch (error) {
      console.error("Error creating agent:", error);
      return res.status(500).json({
        error: sanitizeError(error),
      });
    }
  }
);

/**
 * @swagger
 * /agent/{agent_id}/build:
 *   post:
 *     $ref: '#/paths/~1agent~1{agent_id}~1build/post'
 */
router.post(
  "/:agent_id/build",
  [
    // Validate that agent_id is a valid Mongo ID
    param("agent_id").isMongoId().withMessage("Invalid agent_id"),
  ],
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }
    try {
      const MAX_RETRIES = 3;
      const RETRY_DELAY = 1000; // 1 second

      const { agent_id } = req.params;

      // Verify agent exists
      const agent = await AgentModel.findById(agent_id);
      if (!agent) {
        return res.status(404).json({ error: "Agent not found" });
      }

      // Start document processing pipeline
      await DocumentProcessingController.process_documents(agent_id);

      try {
        // Get agent's raw files
        const raw_files = await agentFileService.get_agent_raw_files(agent_id);

        // Process each unprocessed file
        const files_to_process = raw_files
          .filter((file) => file.processing_status === "pending")
          .map((file) => ({
            file_url: file.file_url,
            retries: 0,
          }));

        // Process files sequentially with retries
        const processing_results = [];

        for (const file of files_to_process) {
          let success = false;
          while (!success && file.retries < MAX_RETRIES) {
            try {
              const result = await agentFileService.process_and_upload_file(
                agent_id,
                file.file_url
              );
              processing_results.push({
                file_url: file.file_url,
                status: "success",
                result,
              });
              success = true;
            } catch (error) {
              file.retries++;
              if (file.retries < MAX_RETRIES) {
                await new Promise((resolve) =>
                  setTimeout(resolve, RETRY_DELAY * file.retries)
                );
              } else {
                processing_results.push({
                  file_url: file.file_url,
                  status: "failed",
                  error: error.message,
                  retries: file.retries,
                });
              }
            }
          }
        }

        // Send response before potentially long-running operations
        res.status(202).json({
          agent_id,
          status: "processing",
          total_files: files_to_process.length,
          processed: processing_results.filter((r) => r.status === "success")
            .length,
          failed: processing_results.filter((r) => r.status === "failed")
            .length,
        });

        // Continue with background processing if needed
        if (processing_results.some((r) => r.status === "success")) {
          try {
            // Update assistant with new files (if any successful uploads)
            const successfulFiles = processing_results
              .filter((r) => r.status === "success")
              .map((r) => r.result.file_id);

            if (successfulFiles.length > 0 && agent.openai_assistant_id) {
              const { vector_store } =
                await openAIService.add_files_to_assistant({
                  assistant_id: agent.openai_assistant_id,
                  file_ids: successfulFiles,
                  vector_store_name: `${agent.name}_vector_store`,
                });

              await AgentModel.findByIdAndUpdate(agent_id, {
                openai_vector_store_id: vector_store.id,
                queue_status: "completed",
              });
            }
          } catch (error) {
            console.error("Error updating assistant:", error);
            await AgentModel.findByIdAndUpdate(agent_id, {
              queue_status: "failed",
              error_logs: [
                {
                  error_message: error.message,
                  error_type: "AssistantUpdateError",
                },
              ],
            });
          }
        }
      } catch (error) {
        // If error occurs after response is sent, just log it
        console.error("Error in background processing:", error);
        await AgentModel.findByIdAndUpdate(agent_id, {
          queue_status: "failed",
          error_logs: [
            {
              error_message: error.message,
              error_type: "BackgroundProcessingError",
            },
          ],
        });
      }
    } catch (error) {
      console.error("Error triggering build:", error);
      if (!res.headersSent) {
        res.status(500).json({
          error: sanitizeError(error),
        });
      }
    }
  }
);

/**
 * @swagger
 * /agent/{agent_id}/queue-status:
 *   get:
 *     $ref: '#/paths/~1agent~1{agent_id}~1queue-status/get'
 */
router.get(
  "/:agent_id/queue-status",
  [
    // Validate that agent_id is a valid Mongo ID
    param("agent_id").isMongoId().withMessage("Invalid agent_id"),
  ],
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }
    try {
      const status = await DocumentProcessingController.get_agent_queue(
        req.params.agent_id
      );
      res.json(status);
    } catch (error) {
      if (error.status === 404) {
        res.status(404).json({ error: "Queue status not found" });
      } else {
        res.status(500).json({
          error: sanitizeError(error),
        });
      }
    }
  }
);

/**
 * @swagger
 * /agent/{agent_id}:
 *   delete:
 *     tags: [AI Pipeline]
 *     summary: Mark an agent as removed
 *     description: Updates the agent's is_removed field to true
 *     parameters:
 *       - in: path
 *         name: agent_id
 *         required: true
 *         description: ID of the agent to mark as removed
 *         schema:
 *           type: string
 */
router.delete(
  "/:agent_id",
  [param("agent_id").isMongoId().withMessage("Invalid agent_id")],
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    try {
      const agent = await AgentModel.findByIdAndUpdate(
        req.params.agent_id,
        { is_removed: true },
        { new: true }
      );

      if (!agent) {
        return res.status(404).json({ error: "Agent not found" });
      }

      res.json({ message: "Agent successfully deleted" });
    } catch (error) {
      console.error("Error removing agent:", error);
      res.status(500).json({
        error: sanitizeError(error),
      });
    }
  }
);

export default router;
