import express from "express";
import { body, param, validationResult } from "express-validator";
import { ChatController } from "../controllers/chatController.js";
import { audioUpload } from "../../config/upload.js";
import {
  transcriptionLimiter,
  synthesisLimiter,
} from "../middleware/speechRateLimit.js";

// Middleware to handle validation errors
const handleValidationErrors = (req, res, next) => {
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).json({ success: false, errors: errors.array() });
  }
  next();
};

const router = express.Router();

/**
 * @swagger
 * /agent/chat/conversations:
 *   post:
 *     tags:
 *       - AI Pipeline
 *     summary: Create a new conversation
 *     description: Creates a new conversation thread with an AI assistant
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - agent_id
 *               - user_id
 *               - initial_message
 *             properties:
 *               agent_id:
 *                 type: string
 *                 description: MongoDB ID of the AI agent
 *               user_id:
 *                 type: string
 *                 description: MongoDB ID of the user
 *               initial_message:
 *                 type: string
 *                 description: First message to start the conversation
 *     responses:
 *       201:
 *         description: Conversation created successfully
 *       500:
 *         description: Server error
 */
router.post(
  "/conversations",
  [
    body("agent_id").isMongoId().withMessage("Invalid agent_id"),
    body("user_id").isMongoId().withMessage("Invalid user_id"),
    body("initial_message")
      .notEmpty()
      .withMessage("Initial message is required"),
  ],
  handleValidationErrors,
  ChatController.createConversation
);

/**
 * @swagger
 * /agent/chat/conversations/message:
 *   post:
 *     tags:
 *       - AI Pipeline
 *     summary: Send a message in a conversation
 *     description: Sends a message to an existing conversation and gets the AI response with both text and synthesized speech
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - conversation_id
 *               - message
 *             properties:
 *               conversation_id:
 *                 type: string
 *                 description: MongoDB ID of the conversation
 *               message:
 *                 type: string
 *                 description: Message to send
 *     responses:
 *       200:
 *         description: Message sent and response received successfully
 *       404:
 *         description: Conversation not found
 *       500:
 *         description: Server error
 */
router.post(
  "/conversations/message",
  [
    body("conversation_id").isMongoId().withMessage("Invalid conversation_id"),
    body("message").notEmpty().withMessage("Message is required"),
  ],
  handleValidationErrors,
  ChatController.sendMessage
);

/**
 * @swagger
 * /agent/chat/conversations/{conversation_id}:
 *   get:
 *     tags:
 *       - AI Pipeline
 *     summary: Get conversation history
 *     description: Retrieves the full history of a conversation
 *     parameters:
 *       - in: path
 *         name: conversation_id
 *         required: true
 *         schema:
 *           type: string
 *         description: MongoDB ID of the conversation
 *     responses:
 *       200:
 *         description: Conversation retrieved successfully
 *       404:
 *         description: Conversation not found
 *       500:
 *         description: Server error
 */
router.get(
  "/conversations/:conversation_id",
  [param("conversation_id").isMongoId().withMessage("Invalid conversation_id")],
  handleValidationErrors,
  ChatController.getConversation
);

/**
 * @swagger
 * /agent/chat/speech/transcribe:
 *   post:
 *     tags:
 *       - AI Pipeline
 *     summary: Transcribe audio to text
 *     description: Uses OpenAI Whisper model to convert speech to text
 *     requestBody:
 *       required: true
 *       content:
 *         multipart/form-data:
 *           schema:
 *             type: object
 *             required:
 *               - file
 *             properties:
 *               file:
 *                 type: string
 *                 format: binary
 *                 description: Audio file (mp3, wav, or webm - max 25MB)
 *     responses:
 *       200:
 *         description: Audio transcribed successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: true
 *                 data:
 *                   type: object
 *                   properties:
 *                     text:
 *                       type: string
 *                       description: Transcribed text from the audio
 *       400:
 *         description: Bad request
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: No audio file provided OR Invalid file type
 *       429:
 *         description: Too Many Requests
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Too many transcription requests. Please try again later.
 *       500:
 *         description: Server error
 */
router.post(
  "/speech/transcribe",
  transcriptionLimiter,
  audioUpload.single("file"),
  ChatController.transcribeAudio
);

/**
 * @swagger
 * /agent/chat/speech/synthesize:
 *   post:
 *     tags:
 *       - AI Pipeline
 *     summary: Convert text to speech
 *     description: Uses OpenAI TTS to convert text to speech with various voice options
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - text
 *             properties:
 *               text:
 *                 type: string
 *                 description: Text to convert to speech
 *                 example: Hello, how can I help you today?
 *               voice:
 *                 type: string
 *                 enum: [alloy, echo, fable, onyx, nova, shimmer]
 *                 default: alloy
 *                 description: Voice style to use for speech synthesis
 *     responses:
 *       200:
 *         description: Speech synthesized successfully
 *         content:
 *           audio/mpeg:
 *             schema:
 *               type: string
 *               format: binary
 *               description: MP3 audio file containing synthesized speech
 *       400:
 *         description: Bad request
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Text is required OR Invalid voice option
 *       429:
 *         description: Too Many Requests
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   example: Too many speech synthesis requests. Please try again later.
 *       500:
 *         description: Server error
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                   example: false
 *                 error:
 *                   type: string
 *                   description: Error message from OpenAI API
 */
router.post(
  "/speech/synthesize",
  synthesisLimiter,
  ChatController.synthesizeSpeech
);

/**
 * @swagger
 * /agent/chat/conversations/unified-message:
 *   post:
 *     tags:
 *       - AI Pipeline
 *     summary: Send a unified message (text or voice) in a conversation
 *     description: Accepts either text or voice input (but not both) and returns both text and synthesized speech response
 *     requestBody:
 *       required: true
 *       content:
 *         multipart/form-data:
 *           schema:
 *             type: object
 *             required:
 *               - conversation_id
 *             properties:
 *               conversation_id:
 *                 type: string
 *                 description: MongoDB ID of the conversation
 *               text:
 *                 type: string
 *                 description: Text message (mutually exclusive with audio file)
 *               audio:
 *                 type: string
 *                 format: binary
 *                 description: Audio file upload (mp3, wav, or webm - max 25MB, mutually exclusive with text)
 *     responses:
 *       200:
 *         description: Message processed successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 success:
 *                   type: boolean
 *                 data:
 *                   type: object
 *                   properties:
 *                     message:
 *                       type: string
 *                       description: The AI assistant's text response
 *                     audio:
 *                       type: string
 *                       description: Base64 encoded MP3 audio of the AI's response
 *                     conversation:
 *                       type: object
 *                       description: Updated conversation object with message history
 *       400:
 *         description: Invalid input - must provide exactly one of text or audio
 *       404:
 *         description: Conversation not found
 *       500:
 *         description: Server error
 */
router.post(
  "/conversations/unified-message",
  audioUpload.single("audio"),
  ChatController.handleUnifiedMessage
);

export default router;
