import fs from 'fs/promises';
import path from 'path';

/**
 * Service for handling file operations in the AI Pipeline
 */
export class FileService {
    /**
     * @param {string} uploadDir Directory for file uploads
     */
    constructor(uploadDir) {
        this.uploadDir = uploadDir;
    }

    /**
     * Initialize the file service and ensure upload directory exists
     * @returns {Promise<void>}
     * @throws {Error} If directory creation or permissions test fails
     */
    async initialize() {
        try {
            // Create uploads directory if it doesn't exist
            await fs.mkdir(this.uploadDir, { recursive: true });
            
            // Test write permissions with a temporary file
            const testFile = path.join(this.uploadDir, '.test');
            await fs.writeFile(testFile, '');
            await fs.unlink(testFile);
        } catch (error) {
            throw new Error(`Failed to initialize file service: ${error.message}`);
        }
    }

    /**
     * Get the configured upload directory
     * @returns {string} Upload directory path
     */
    getUploadDir() {
        return this.uploadDir;
    }

    /**
     * Clean up any temporary files
     * @returns {Promise<void>}
     */
    async cleanup() {
        try {
            // Add specific cleanup logic here if needed
            // For example, removing temporary processing files
            const files = await fs.readdir(this.uploadDir);
            const tempFiles = files.filter(file => file.startsWith('.temp_'));
            
            await Promise.all(
                tempFiles.map(file => 
                    fs.unlink(path.join(this.uploadDir, file))
                )
            );
        } catch (error) {
            console.error('Error during file service cleanup:', error);
            throw error;
        }
    }
}