// server.js
import express from "express";
import cors from "cors";
import dotenv from "dotenv";
import helmet from "helmet";
import compression from "compression";
import mongoose from "mongoose";
import path from "path";
import os from "os";
import { fileURLToPath } from "url";
import meetingRoutes from "./routes/meetingRoutes.js";

// Swagger
import swaggerUi from "swagger-ui-express";
import { swaggerSpecs } from "./docs/swagger.js";

// Config & Middleware
import { connectDB } from "./config/database.js";
import { rateLimiter } from "./middleware/rateLimit.js";
import {
  authenticateOrigin,
  authenticateUserAgent,
} from "./middleware/auth.js";

// Routes
import authRoutes from "./routes/auth.js";
import currenciesRoutes from "./routes/currencies.js";
import languagesRoutes from "./routes/language.js";
import uploadRoutes from "./routes/upload.js";
import exhibitionsRoutes from "./routes/exhibitions.js";
import boothsRoutes from "./routes/booths.js";
import filesRoutes from "./routes/file.js";
import paymentsRoutes from "./routes/payment.js";
import invoiceRoutes from "./routes/invoice.routes.js";
import testTokenRoute from "./routes/testToken.js";

// AI
import AIPipeline from "./AI-PipeLine/index.js";

// Utils
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// Load env
dotenv.config();

// Init app
const app = express();
const PORT =
  process.env.NODE_ENV === "development"
    ? process.env.PORT_DEV || 7000
    : process.env.PORT || 7000;

// Connect DB
await connectDB();
console.log("✅ Connected to MongoDB");

// Global Middleware
app.use(helmet());
app.use(express.json({ limit: "10kb" }));
app.use(express.urlencoded({ extended: true, limit: "10kb" }));
app.use(compression());
app.use(rateLimiter);

// 🌐 CORS — آزاد برای همه دامنه‌ها
app.use(
  cors({
    origin: "*", // همه دامنه‌ها
    methods: ["GET","HEAD","PUT","PATCH","POST","DELETE"],
    allowedHeaders: ["Content-Type","Authorization","X-Requested-With"],
    credentials: false, // اگر true شود، origin نمی‌تواند '*'
    preflightContinue: false,
    optionsSuccessStatus: 204,
  })
);

// Swagger UI
app.use(
  "/api-docs",
  authenticateOrigin,
  authenticateUserAgent,
  swaggerUi.serve,
  swaggerUi.setup(swaggerSpecs)
);

// Health Check
app.get("/health", authenticateOrigin, authenticateUserAgent, (req, res) => {
  res.status(200).json({
    status: "healthy",
    uptime: process.uptime(),
    timestamp: new Date().toISOString(),
  });
});

app.get("/ping", authenticateOrigin, authenticateUserAgent, (req, res) => {
  const { timestamp } = req.query;
  if (timestamp) {
    const inputTime = parseInt(timestamp, 10);
    if (isNaN(inputTime)) {
      return res.status(400).json({ message: "Invalid timestamp" });
    }
    const currentTime = Date.now();
    const difference = currentTime - inputTime;
    return res.status(200).json({ message: "pong", ms: difference });
  }
  return res.status(200).json({ message: "pong" });
});

const iconsPath = path.join(process.cwd(), "src", "AI-PipeLine", "Icon_currencies");

// Routes
app.use("/user", authRoutes);
app.use("/currencies", currenciesRoutes);
app.use("/languages", languagesRoutes);
app.use("/upload", uploadRoutes);
app.use("/exhibitions", exhibitionsRoutes);
app.use("/exhibitions/booths", boothsRoutes);
app.use("/booths", boothsRoutes);
app.use("/files", filesRoutes);
app.use("/payments", paymentsRoutes);
app.use("/invoices", invoiceRoutes);
app.use("/icons", express.static(iconsPath));

app.use("/test", testTokenRoute);
app.use("/meeting", meetingRoutes);

// 🧩 Static uploads (برای دسترسی داخلی)
app.use(
  "/uploads",
  express.static(path.join(__dirname, "..", process.env.UPLOAD_DIR || "uploads"))
);

// 🧩 Static for CDN access (cdn.webcomtower.ir/...)
app.use(
  "/",
  express.static(path.join(__dirname, "..", process.env.UPLOAD_DIR || "uploads"))
);

// 🧠 AI Pipeline Init
await AIPipeline.initialize(app, {
  uploadDir: path.join(__dirname, "..", process.env.UPLOAD_DIR || "uploads"),
  openaiApiKey: process.env.OPENAI_API_KEY,
});
console.log("✅ AI Pipeline initialized");

// Global Error Handler
app.use((err, req, res, next) => {
  console.error("Error:", err);
  res.status(err.status || 500).json({
    error:
      process.env.NODE_ENV === "production"
        ? "Internal Server Error"
        : err.message,
    status: err.status || 500,
    stack: process.env.NODE_ENV === "production" ? undefined : err.stack,
  });
});

// Start server
const server = app.listen(PORT, "0.0.0.0", () => {
  console.log(`🚀 Server running at http://${getLocalIP()}:${PORT}`);
});

// Graceful shutdown
async function shutdown(signal) {
  console.log(`\n${signal} received. Starting graceful shutdown...`);
  const shutdownTimeout = setTimeout(() => {
    console.error("Forcefully shutting down...");
    process.exit(1);
  }, 30000);

  server.close(async () => {
    try {
      clearTimeout(shutdownTimeout);
      await AIPipeline.cleanup();
      await mongoose.connection.close();
      console.log("✅ Graceful shutdown complete");
      process.exit(0);
    } catch (error) {
      console.error("❌ Error during shutdown:", error);
      process.exit(1);
    }
  });
}

process.on("SIGTERM", () => shutdown("SIGTERM"));
process.on("SIGINT", () => shutdown("SIGINT"));

// Utility: Get local IP address
function getLocalIP() {
  const interfaces = os.networkInterfaces();
  for (const name of Object.keys(interfaces)) {
    for (const iface of interfaces[name]) {
      if (iface.family === "IPv4" && !iface.internal) {
        return iface.address;
      }
    }
  }
  return "localhost";
}
