import fs from 'fs/promises';
import path from 'path';
import dotenv from 'dotenv';
import axios from 'axios';
import FormData from 'form-data';
import FileModel from '../models/file.js';
import { BoughtenBoothsModel } from '../models/boughten_booths.js';
import { AgentModel } from '../AI-PipeLine/models/agent.js';

dotenv.config();

const UPLOAD_ROOT = process.env.UPLOAD_DIR || 'uploads';
const CDN_BASE = process.env.CDN_BASE || 'http://192.168.10.50:6015';

export const uploadFile = async (req, res) => {
  try {
    const file = req.validatedFile;
    const { ref, ref_id, exhibition_id } = req.body;
    const userId = req.user?.id || null;

    if (!ref) {
      return res.status(400).json({ success: false, message: 'فیلد ref الزامی است' });
    }

    let filePath = '';
    let booth = null;

    const boothRefs = ['posters', 'monitors', 'stands', 'video_walls', 'agents', 'logo'];

    if (boothRefs.includes(ref)) {
      if (!ref_id) {
        return res.status(400).json({ success: false, message: 'ref_id برای این دسته‌بندی لازم است' });
      }

      booth = await BoughtenBoothsModel.findById(ref_id);

      if (!booth) {
        if (!exhibition_id) {
          return res.status(400).json({ success: false, message: 'exhibition_id برای ساخت غرفه جدید لازم است' });
        }
        booth = await BoughtenBoothsModel.create({ exhibition_id });
      }

      if (!booth.exhibition_id) {
        return res.status(400).json({ success: false, message: 'غرفه ساخته شده exhibition_id ندارد' });
      }

      filePath = ref === 'agents'
        ? `exhibitions/${booth.exhibition_id}/boughten_booths/${booth._id}/agents/raw_files`
        : `exhibitions/${booth.exhibition_id}/boughten_booths/${booth._id}/${ref}`;

    } else {
      switch (ref) {
        case 'organizations':
          if (!ref_id) return res.status(400).json({ success: false, message: 'ref_id لازم است برای organizations' });
          filePath = `organizations/${ref_id}`;
          break;
        case 'users':
          if (!userId) return res.status(401).json({ success: false, message: 'User ID missing from token' });
          filePath = `users/${userId}`;
          break;
        case 'exhibitions':
          if (!ref_id) return res.status(400).json({ success: false, message: 'ref_id لازم است برای exhibitions' });
          filePath = `exhibitions/${ref_id}`;
          break;
        case 'boughten_booths':
          if (!ref_id) return res.status(400).json({ success: false, message: 'ref_id لازم است برای boughten_booths' });
          booth = await BoughtenBoothsModel.findById(ref_id);
          if (!booth || !booth.exhibition_id) {
            return res.status(400).json({ success: false, message: 'غرفه یا نمایشگاه یافت نشد' });
          }
          filePath = `exhibitions/${booth.exhibition_id}/boughten_booths/${booth._id}`;
          break;
        default:
          return res.status(400).json({ success: false, message: 'نوع ref نامعتبر است' });
      }
    }

    // امن‌سازی نام فایل
    const originalName = Buffer.from(file.originalname, 'binary').toString('utf8');
    const safeName = `raw_file_${originalName}`.replace(/[<>:"/\\|?*]/g, '');

    // مسیر کامل دایرکتوری
    const fullDir = path.join(process.cwd(), UPLOAD_ROOT, filePath);

    // حذف فایل‌های قبلی در این پوشه
    try {
      const existingFiles = await fs.readdir(fullDir);
      for (const f of existingFiles) {
        const fullPath = path.join(fullDir, f);
        await fs.unlink(fullPath);
      }
    } catch (err) {
      // اگر پوشه وجود نداشت، نیازی به کاری نیست
    }

    // ایجاد پوشه در صورت نیاز و ذخیره فایل جدید
    await fs.mkdir(fullDir, { recursive: true });
    const finalPath = path.join(fullDir, safeName);
    await fs.writeFile(finalPath, file.buffer);

    // ارسال به CDN
    const form = new FormData();
    form.append('file', file.buffer, {
      filename: safeName,
      contentType: file.mimetype,
    });
    form.append('ref', ref);
    form.append('ref_id', ref === 'users' ? userId : ref_id);

    let cdnUrl = '';
    try {
      const cdnRes = await axios.post(`${CDN_BASE}/upload`, form, {
        headers: form.getHeaders(),
      });
      cdnUrl = cdnRes?.data?.data?.file?.url || `${CDN_BASE}/uploads/${filePath}/${safeName}`;
    } catch (err) {
      console.error('❌ Error uploading to CDN:', err?.response?.data || err.message);
      cdnUrl = `${CDN_BASE}/uploads/${filePath}/${safeName}`;
    }

    // ذخیره اطلاعات فایل در دیتابیس
    const fileDoc = await FileModel.create({
      originalName,
      fileName: safeName,
      mimeType: file.mimetype || 'application/octet-stream',
      size: file.size,
      type: file.mimetype || 'application/octet-stream',
      ref,
      ref_id: ref === 'users' ? userId : ref_id,
      path: filePath,
      url: cdnUrl,
      uploadedAt: new Date(),
    });

    // اتصال به agent (اگر لازم باشد)
    if (ref === 'agents') {
      await AgentModel.findByIdAndUpdate(ref_id, {
        $push: { raw_files: { file_id: fileDoc._id } },
      });
    }

    return res.status(201).json({
      success: true,
      message: 'File uploaded successfully',
      data: {
        file: {
          originalName,
          fileName: safeName,
          mimeType: file.mimetype,
          size: file.size,
          url: cdnUrl,
          ref,
          ref_id: ref === 'users' ? userId : ref_id,
          uploadedAt: new Date().toISOString(),
        },
        uploadStats: {
          processingTime: 1234,
          status: 'completed',
        },
      },
    });
  } catch (error) {
    console.error('❌ Upload Error:', error);
    return res.status(500).json({
      success: false,
      message: 'File upload failed',
      error: error.message,
    });
  }
};
export const deleteFilesByRefAndBooth = async (req, res) => {
  try {
    const { booth_id, ref } = req.query;

    if (!booth_id || !ref) {
      return res.status(400).json({
        success: false,
        message: 'booth_id and ref are required',
      });
    }

    const booth = await BoughtenBoothsModel.findById(booth_id);
    if (!booth || !booth.exhibition_id) {
      return res.status(404).json({
        success: false,
        message: 'booth_not_found',
      });
    }

    const folderPath = path.join(
      process.cwd(),
      UPLOAD_ROOT,
      'exhibitions',
      booth.exhibition_id.toString(),
      'boughten_booths',
      booth._id.toString(),
      ref
    );

    await fs.rm(folderPath, { recursive: true, force: true });

    return res.status(200).json({
      success: true,
      message: 'Files deleted successfully',
    });
  } catch (error) {
    console.error('❌ Delete Error:', error);
    return res.status(500).json({
      success: false,
      message: 'Failed to delete files',
      error: error.message,
    });
  }
};

