// middleware/validateFileUpload.js
import { body, validationResult } from 'express-validator';
import aiFileValidation from '../AI-PipeLine/config/fileValidation.js';
import path from 'path';
import fs from 'fs';
import { imageSize } from 'image-size';

const ALLOWED_REFS = [
  'organizations', 'users', 'exhibitions', 'boughten_booths',
  'video_walls', 'posters', 'logo', 'monitors', 'stands', 'agents'
];

const validateFile = (file, ref) => {
  const errors = [];

  if (!file) {
    errors.push('No file uploaded');
    return { valid: false, errors };
  }

  if (!file.originalname || !file.mimetype || !file.size) {
    errors.push('Invalid file metadata');
    return { valid: false, errors };
  }

  const fileExtension = path.extname(file.originalname).toLowerCase();
  const mimeType = file.mimetype;

  const allowedMime = aiFileValidation.allowedTypes[fileExtension];

  if (!allowedMime) {
    errors.push(`File type ${fileExtension} is not allowed`);
  }

  if (allowedMime !== mimeType) {
    errors.push('File type does not match its extension');
  }

  const maxSize = aiFileValidation.maxFileSizes[mimeType] || 0;
  if (file.size > maxSize) {
    errors.push(`File size exceeds the maximum limit of ${maxSize / (1024 * 1024)}MB`);
  }

  if (file.size === 0) {
    errors.push('File is empty');
  }

  if (file.originalname.length > 255) {
    errors.push('File name is too long');
  }

  if (/[<>:"/\\|?*]/.test(file.originalname)) {
    errors.push('File name contains invalid characters');
  }

  if (file.mimetype.startsWith('image/') && aiFileValidation.allowedDimensions[ref]) {
    try {
      let dimensions;
      if (file.buffer) {
        dimensions = imageSize(file.buffer);
      } else if (file.path) {
        const buffer = fs.readFileSync(file.path);
        dimensions = imageSize(buffer);
      } else {
        errors.push('File buffer or path not found');
      }

      if (dimensions) {
        const expected = aiFileValidation.allowedDimensions[ref];
        if (
          dimensions.width !== expected.width ||
          dimensions.height !== expected.height
        ) {
          errors.push(`Invalid image dimensions: expected ${expected.width}x${expected.height}px, got ${dimensions.width}x${dimensions.height}px`);
        }
      }
    } catch (err) {
      errors.push('Unable to read image dimensions');
    }
  }

  return {
    valid: errors.length === 0,
    errors
  };
};

export const validateFileUpload = [
  body('ref')
    .trim()
    .notEmpty()
    .withMessage('Reference type is required')
    .isIn(ALLOWED_REFS)
    .withMessage('Invalid reference type')
    .escape(),

  body('ref_id')
    .custom((value, { req }) => {
      if (req.body.ref === 'users') return true;
      return value && value.trim() !== '' && /^[a-f\d]{24}$/i.test(value);
    })
    .withMessage('Invalid reference ID format')
    .escape(),

  async (req, res, next) => {
    try {
      const errors = validationResult(req);
      if (!errors.isEmpty()) {
        return res.status(400).json({
          success: false,
          errors: errors.array(),
          message: 'Validation error'
        });
      }

      const fileValidation = validateFile(req.file, req.body.ref);
      if (!fileValidation.valid) {
        return res.status(400).json({
          success: false,
          errors: fileValidation.errors,
          message: 'File validation failed'
        });
      }

      req.validatedFile = {
        ...req.file,
        validated: true,
        validatedAt: new Date().toISOString()
      };

      next();
    } catch (error) {
      console.error('File validation error:', error);
      res.status(500).json({
        success: false,
        message: 'Internal server error during file validation',
        error: process.env.NODE_ENV === 'development' ? error.message : undefined
      });
    }
  }
];
