import express from "express";
import { body, validationResult } from "express-validator";
import { handleSendCode, handleCheckCode } from "../services/auth.js";
import {
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
} from "../middleware/auth.js";
import { isIranianMobilePhone } from "../utils/validation.js";
import { UserModel } from "../models/users.js";

const router = express.Router();

/**
 * @swagger
 * tags:
 *   - name: Authentication
 *     description: Endpoints for user authentication
 */

/**
 * @swagger
 * /user/sendCode:
 *   post:
 *     summary: Send verification code
 *     tags: [Authentication]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               phone:
 *                 type: string
 *                 required: true
 *                 description: Iranian mobile phone number
 *               metaboard_access:
 *                 type: boolean
 *                 required: true
 *                 description: Access flag for metaboard functionality
 *     responses:
 *       200:
 *         description: Code sent successfully and account status
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 data:
 *                   type: null
 *                 message:
 *                   type: string
 *                   enum: ['Created Account', 'Login']
 *       400:
 *         description: Invalid phone number
 *       429:
 *         description: Too many requests, try again later
 *       500:
 *         description: Server error occurred during SMS sending
 */
router.post(
  "/sendCode",
  [
    body("phone")
      .custom(isIranianMobilePhone)
      .withMessage("phone_number_is_not_valid")
      .notEmpty()
      .withMessage("phone_number_cant_be_empty")
      .isLength({ min: 11, max: 11 })
      .withMessage("phone_number_should_be_between_11_and_11_characters")
      .trim()
      .escape(),
    body("metaboard_access")
      .notEmpty()
      .withMessage("metaboard_access_cant_be_empty")
      .isBoolean()
      .withMessage("metaboard_access_should_be_boolean")
      .trim()
      .escape(),
  ],
  authenticateOrigin,
  authenticateUserAgent,
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "validation error",
      });
    }

    const result = await handleSendCode(
      req.body.phone,
      req.body.metaboard_access
    );
    res.status(result.status).json({
      data: result.data,
      message: result.message,
    });
  }
);

/**
 * @swagger
 * /user/checkCode:
 *   post:
 *     summary: Verify code
 *     tags: [Authentication]
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               phone:
 *                 type: string
 *                 description: Iranian mobile phone number
 *               code:
 *                 type: string
 *                 description: Verification code
 *     responses:
 *       200:
 *         description: Code verified successfully
 *       400:
 *         description: Invalid code or phone number
 */
router.post(
  "/checkCode",
  [
    body("phone")
      .notEmpty()
      .withMessage("phone_number_cant_be_empty")
      .isLength({ min: 11, max: 11 })
      .withMessage("phone_number_should_be_11_digits")
      .custom(isIranianMobilePhone)
      .withMessage("phone_number_is_not_valid")
      .trim()
      .escape(),

    body("code")
      .notEmpty()
      .withMessage("otp_code_cant_be_empty")
      .isLength({ min: 4, max: 4 })
      .withMessage("otp_code_should_be_4_digits")
      .isNumeric()
      .withMessage("otp_code_is_not_valid")
      .trim()
      .escape(),
  ],
  authenticateOrigin,
  authenticateUserAgent,
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "validation error",
      });
    }

    const { phone, code } = req.body;

    // چک کردن کد و شماره
    const result = await handleCheckCode(phone, code);

    if (!result.success) {
      // اگر ناموفق بود، جواب رو بفرست
      return res.status(result.status).json({
        data: result.data || null,
        message: result.message,
      });
    }

    // اگر موفق بود و شماره وجود داشت، وضعیت کاربر رو active کن
    try {
      const user = await UserModel.findOne({ phone });

      if (user && user.status !== "active") {
        user.status = "active";
        await user.save();
      }
    } catch (error) {
      // اگر خطایی بود، لاگ کن ولی پاسخ موفق رو بفرست
      console.error("Error updating user status:", error);
    }

    return res.status(result.status).json({
      data: result.data || null,
      message: result.message || "User activated",
    });
  }
);

/**
 * @swagger
 * /user/complete_information:
 *   post:
 *     summary: Complete user information
 *     tags: [Authentication]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             properties:
 *               name:
 *                 type: string
 *                 required: true
 *                 description: User's name
 *               national_code:
 *                 type: string
 *                 required: true
 *                 pattern: '^[0-9]{10}$'
 *                 description: User's national code (10 digits)
 *     responses:
 *       200:
 *         description: Information updated successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 data:
 *                   type: null
 *                 message:
 *                   type: string
 *                   example: "ok"
 *       400:
 *         description: Validation errors for name or national code
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 data:
 *                   type: null
 *                 message:
 *                   type: string
 *                   example: "validation error"
 *       401:
 *         description: User not found or invalid authentication token
 */
router.post(
  "/complete_information",
  [
    body("name")
      .notEmpty()
      .withMessage("name_cant_be_empty")
      .isLength({ min: 3, max: 50 })
      .withMessage("name_should_be_between_3_and_50_characters")
      .trim()
      .escape(),
    body("national_code")
      .isNumeric()
      .withMessage("national_code_is_not_valid")
      .isLength({ min: 10, max: 10 })
      .withMessage("national_code_must_be_10_digits")
      .notEmpty()
      .withMessage("national_code_cant_be_empty")
      .trim()
      .escape(),
  ],
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "validation error",
      });
    }
    await UserModel.updateOne(
      { _id: req.user.id },
      { national_code: req.body.national_code, name: req.body.name }
    );
    res.status(200).json({
      data: null,
      message: "ok",
    });
  }
);

/**
 * @swagger
 * /user/info:
 *   get:
 *     summary: Retrieve authenticated user's information
 *     tags: [Authentication]
 *     security:
 *       - bearerAuth: []
 *     responses:
 *       200:
 *         description: User information retrieved successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 data:
 *                   type: object
 *                   properties:
 *                     name:
 *                       type: string
 *                       description: User's full name
 *                     phone:
 *                       type: string
 *                       description: User's phone number
 *                     national_code:
 *                       type: string
 *                       nullable: true
 *                       description: User's national code (10 digits)
 *                     metaboard_access:
 *                       type: boolean
 *                       description: Flag indicating metaboard access permission
 *                     admin_access:
 *                       type: boolean
 *                       description: Flag indicating admin access permission
 *                     createdAt:
 *                       type: string
 *                       format: date-time
 *                       description: Account creation timestamp
 *                     updatedAt:
 *                       type: string
 *                       format: date-time
 *                       description: Last update timestamp
 *                 message:
 *                   type: string
 *                   example: "ok"
 *       401:
 *         description: Unauthorized access - User not found or invalid token
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 data:
 *                   type: null
 *                 message:
 *                   type: string
 *                   example: "Unauthorized"
 */
router.get(
  "/info",
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "validation error",
      });
    }

    const user = req.user;

    res.status(200).json({
      data: {
        _id: user._id, // 👈 اضافه شده
        name: user.name,
        phone: `${user.phone.slice(0, 4)}***${user.phone.slice(-4)}`,
        national_code: user.national_code
          ? `${user.national_code.slice(0, 3)}***${user.national_code.slice(-4)}`
          : null,
        metaboard_access: user.metaboard_access,
        admin_access: user.admin_access,
        createdAt: user.createdAt,
        updatedAt: user.updatedAt,
      },
      message: "ok",
    });
  }
);

export default router;
