  // booths.js
  import express from "express";
  import { body, validationResult, param } from "express-validator";
  import {
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
    authenticateUserCompletedInformation,
    authenticateCurrency,
    authenticateOwnerBooth,
    authenticateErrors,
  } from "../middleware/auth.js";
  import { BoothEnumsModel } from "../models/booth_enums.js";
  import { ExhibitionsModel } from "../models/exhibitions.js";
  import { FileModel } from "../models/file.js";
  import { ColorEnumsModel } from "../models/color_enums.js";
  import { BoughtenBoothsModel } from "../models/boughten_booths.js";
  import { PricesModel } from "../models/prices.js";
  import { CurrenciesModel } from "../models/currencies.js";
  import { InvoiceModel } from "../models/invoice.js";
  import { PaymentMethodModel } from "../models/payment_method.js";
  import { UserModel } from "../models/users.js";
  import { VideoWallsModel } from "../models/video_walls.js";
  import { CreateVideoWalls } from "../middleware/video_walls.js";
  import fs from "fs/promises";
  import path from "path";

  const router = express.Router();

  const checkValidation = (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      res.status(400).json({ data: null, errors: errors.array(), message: "Validation error" });
      return true;
    }
    return false;
  };

  router.get("/details/exhibition", [
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
    authenticateUserCompletedInformation,
    authenticateCurrency,
  ], async (req, res) => {
    if (checkValidation(req, res)) return;
    const exhibitions = await ExhibitionsModel.find({ is_removed: false, end_date: { $gt: Date.now() } });
    const enriched = await Promise.all(exhibitions.map(async (ex) => {
      const price = await PricesModel.findOne({ ref: "exhibitions", ref_id: ex._id, currency_id: req.currency._id, is_removed: false });
      const currency = await CurrenciesModel.findById(req.currency._id);
      return { ...ex.toObject(), price_per_year: { price, currency } };
    }));
    res.status(200).json({ data: enriched, message: "ds" });
  });

  router.get("/details/colors", [
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
    authenticateUserCompletedInformation,
  ], async (req, res) => {
    if (checkValidation(req, res)) return;
    const colors = await ColorEnumsModel.find({ is_removed: false });
    res.status(200).json({ data: colors, message: "ds" });
  });

  router.get("/details/enum/:exhibition_id", [
    param("exhibition_id").isMongoId().trim().escape(),
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
    authenticateUserCompletedInformation,
    authenticateCurrency,
  ], async (req, res) => {
    if (checkValidation(req, res)) return;
    const { exhibition_id } = req.params;
    const booth_enums = await BoothEnumsModel.find({ exhibition_id: exhibition_id }).sort({ index: 1 });
    const enriched = await Promise.all(booth_enums.map(async (enumItem) => {
      const boughtCount = await BoughtenBoothsModel.countDocuments({ booth_enum_id: enumItem._id, end_date: { $gt: Date.now() } });
      const price = await PricesModel.findOne({ ref: "booth_enums", ref_id: enumItem._id, currency_id: req.currency._id, is_removed: false });
      const currency = await CurrenciesModel.findById(req.currency._id);
      return {
        ...enumItem.toObject(),
        price_per_year: { price, currency },
        remaining: enumItem.floor_1 + enumItem.floor_2 + enumItem.floor_3 - boughtCount,
      };
    }));
    res.status(200).json({ data: enriched, message: "ds" });
  });

 router.post("/create", [
  body("name").isString().isLength({ min: 3, max: 50 }).trim().escape(),
  body("creator_name").isString().isLength({ min: 3, max: 50 }).trim().escape(),
  body("description").isString().isLength({ min: 10, max: 500 }).trim().escape(),
  body("exhibition_id").isMongoId().trim().escape(),
  body("booth_enum_id").optional().isMongoId().trim().escape(), // ← اختیاری شد
  body("color_enum_id").isMongoId().trim().escape(),
  body("owner").isString().trim().escape(),
  body("owner_id").isMongoId().withMessage("owner_id_is_not_valid"),

  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  authenticateCurrency,
], async (req, res) => {
  if (checkValidation(req, res)) return;

  const { booth_enum_id, color_enum_id, exhibition_id, owner, owner_id } = req.body;

  const [booth_enum, colorEnumExists, exhibitionExists] = await Promise.all([
    booth_enum_id ? BoothEnumsModel.findById(booth_enum_id) : Promise.resolve(true),
    ColorEnumsModel.exists({ _id: color_enum_id }),
    ExhibitionsModel.exists({ _id: exhibition_id }),
  ]);

  if (!colorEnumExists || !exhibitionExists || booth_enum === null) {
    return res.sendStatus(404);
  }

  if (owner !== "users" || !(await UserModel.exists({ _id: owner_id }))) {
    return res.status(406).json({ message: "owner_not_acceptable" });
  }

  const bbm = await BoughtenBoothsModel.create({
    ...req.body,
    owner_id: owner_id,
    path: `/exhibitions/${exhibition_id}/boughten_booths/`,
  });

  bbm.path += bbm._id;
  await bbm.save();

  const price = await PricesModel.findOne({ currency_id: req.currency._id, is_removed: false });
  const method = await PaymentMethodModel.findOne({ currency_id: req.currency._id, is_removed: false });

  if (!price || !method) {
    return res.status(404).json({ message: "payment info not available" });
  }

  const invoice = await InvoiceModel.create({
    ref: "boughten_booths",
    ref_id: bbm._id,
    amount: price.price,
    payment_method_id: method._id,
  });

  res.status(200).json({ data: invoice, message: "ds" });
});



router.get("/video_walls/:boughten_booth_id", [
  param("boughten_booth_id").isMongoId().trim().escape(),
  authenticateErrors,
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  authenticateOwnerBooth,
], async (req, res) => {
  const video_walls = await VideoWallsModel.find({ ref: "boughten_booths", ref_id: req.boughten_booth._id });
  res.status(200).json({ data: video_walls });
});

router.post("/video_walls/:boughten_booth_id", [
  param("boughten_booth_id").isMongoId().trim().escape(),
  body("url").isURL(),
  body("file_id").isMongoId(),
  authenticateErrors,
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  authenticateOwnerBooth,
], async (req, res) => {
  const { url, file_id } = req.body;
  const VideoWall = await CreateVideoWalls("boughten_booths", req.boughten_booth._id, url, file_id);
  res.status(200).json({ data: VideoWall });
});
/**
 * @swagger
 * /exhibitions/booths/all:
 *   post:
 *     summary: دریافت لیست غرفه‌ها به‌همراه فایل‌ها
 *     tags: [Booths]
 *     security:
 *       - bearerAuth: []
 *     requestBody:
 *       required: true
 *       content:
 *         application/json:
 *           schema:
 *             type: object
 *             required:
 *               - owner_id
 *             properties:
 *               owner_id:
 *                 type: string
 *                 example: "60f7b6d3c5e4a911b8a4a3a1"
 *     responses:
 *       200:
 *         description: لیست غرفه‌ها با فایل‌هایشان
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 data:
 *                   type: array
 *                   items:
 *                     type: object
 *                 message:
 *                   type: string
 *                   example: "ارسال شد"
 *       400:
 *         description: ورودی ناقص یا نامعتبر
 *       500:
 *         description: خطای داخلی سرور
 */

router.post(
  "/all",
  [
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
    authenticateUserCompletedInformation,
  ],
  async (req, res) => {
    try {
      const { owner_id } = req.body;
      if (!owner_id)
        return res.status(400).json({ message: "owner_id مشکل دارد" });

      // گرفتن booths و populate کردن رنگ
      const booths = await BoughtenBoothsModel.find({
        owner_id,
        is_removed: { $ne: true },
      })
        .populate("color_enum_id", "index hue name is_removed")
        .lean();

      const basePath = path.join(process.cwd(), "uploads");
      const fileTypes = ["posters", "video_walls", "monitors", "stands"];

      for (const booth of booths) {
        const boothPath = path.join(
          basePath,
          "exhibitions",
          booth.exhibition_id.toString(),
          "boughten_booths",
          booth._id.toString()
        );

        const filesByType = {};

        // خواندن فایل‌های هر نوع
        for (const type of fileTypes) {
          try {
            const files = await fs.readdir(path.join(boothPath, type));
            filesByType[type] = files.map(
              (f) =>
                `http://192.168.10.50:6015/exhibitions/${booth.exhibition_id}/boughten_booths/${booth._id}/${type}/${f}`
            );
          } catch {
            filesByType[type] = [];
          }
        }

        // خواندن فایل لوگو
        try {
          const logoDir = path.join(boothPath, "logo");
          const logoFiles = await fs.readdir(logoDir);
          if (logoFiles.length > 0) {
            filesByType["logo"] = `http://192.168.10.50:6015/exhibitions/${booth.exhibition_id}/boughten_booths/${booth._id}/logo/${logoFiles[0]}`;
          } else {
            filesByType["logo"] = null;
          }
        } catch {
          filesByType["logo"] = null;
        }

        // خواندن فایل‌های agents
        const agentFiles = {};
        for (const sub of ["raw_files", "processed_files"]) {
          try {
            const agentList = await fs.readdir(
              path.join(boothPath, "agents", sub)
            );
            agentFiles[sub] = agentList.map(
              (f) =>
                `http://192.168.10.50:6015/exhibitions/${booth.exhibition_id}/boughten_booths/${booth._id}/agents/${sub}/${f}`
            );
          } catch {
            agentFiles[sub] = [];
          }
        }

        booth.files = { ...filesByType, agents: agentFiles };
      }

      res.status(200).json({ data: booths, message: "ارسال شد" });
    } catch (err) {
      console.error("❌ خطا:", err);
      res.status(500).json({ message: "خطای داخلی سرور" });
    }
  }
);
import mongoose from 'mongoose';


router.post("/update_booth", [
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
], async (req, res) => {
  try {
    const { booth_id, name, color_enum_id } = req.body;

    if (!booth_id) {
      return res.status(400).json({ message: "booth_id الزامی است" });
    }

    const updateFields = {};
    if (name) updateFields.name = name;
    if (color_enum_id) updateFields.color_enum_id = color_enum_id;

    if (Object.keys(updateFields).length === 0) {
      return res.status(400).json({ message: "حداقل یکی از فیلدهای name یا color_enum_id باید ارسال شود" });
    }

    const updated = await BoughtenBoothsModel.findOneAndUpdate(
      { _id: booth_id, is_removed: { $ne: true } },
      updateFields,
      { new: true }
    ).populate("color_enum_id", "index hue name is_removed");

    if (!updated) {
      return res.status(404).json({ message: "غرفه‌ای با این مشخصات پیدا نشد" });
    }

    let color = null;
    if (updated.color_enum_id) {
      const { _id, ...rest } = updated.color_enum_id;
      color = rest;
    }

    res.status(200).json({
      message: "غرفه با موفقیت بروزرسانی شد",
      data: {
        _id: updated._id,
        owner_id: updated.owner_id,
        name: updated.name,
        color: color
      }
    });
  } catch (err) {
    console.error("❌ خطا در بروزرسانی غرفه:", err);
    res.status(500).json({ message: "خطای داخلی سرور" });
  }
});




router.get("/all_owner_id", [
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
], async (req, res) => {
  try {
    // دریافت همه غرفه‌های فعال
    const allBooths = await BoughtenBoothsModel.find({
      is_removed: { $ne: true }
    })
      .populate("color_enum_id", "index hue name is_removed")
      .lean();

    // ساختن خروجی برای هر غرفه، حتی اگر owner_id تکراری باشد
    const result = allBooths.map(booth => {
      let color = null;
      if (booth.color_enum_id) {
        const { _id, ...rest } = booth.color_enum_id;
        color = rest;
      }

      return {
        owner_id: booth.owner_id,
        name: booth.name,
        _id: booth._id,
        color: color
      };
    });

    res.status(200).json({ data: result, message: "ارسال شد" });
  } catch (err) {
    console.error("❌ خطا:", err);
    res.status(500).json({ message: "خطای داخلی سرور" });
  }
});


/**
 * @swagger
 * /booths/full-info/{booth_id}:
 *   get:
 *     summary: دریافت تمام اطلاعات مرتبط با یک غرفه شامل فایل‌ها، ویدیو، نمایشگاه، کاربر و سازمان
 *     tags: [Booths]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: booth_id
 *         required: true
 *         schema:
 *           type: string
 *         description: شناسه غرفه خریداری‌شده
 *     responses:
 *       200:
 *         description: اطلاعات کامل غرفه
 *       404:
 *         description: داده‌ای یافت نشد
 */
// src/routes/booths.js (یا هر فایل روت مربوطه)


/**
 * @swagger
 * /booths/full-info/{booth_id}:
 *   get:
 *     summary: دریافت تمام اطلاعات مرتبط با یک غرفه شامل فایل‌ها، ویدیو، نمایشگاه، کاربر و سازمان
 *     tags: [Booths]
 *     security:
 *       - bearerAuth: []
 *     parameters:
 *       - in: path
 *         name: booth_id
 *         required: true
 *         schema:
 *           type: string
 *         description: شناسه غرفه خریداری‌شده
 *     responses:
 *       200:
 *         description: اطلاعات کامل غرفه
 *       404:
 *         description: داده‌ای یافت نشد
 */
router.get(
  "/full-info/:booth_id",
  [
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
  ],
  async (req, res) => {
    try {
      const { booth_id } = req.params;

      const booth = await BoughtenBoothsModel.findById(booth_id);
      if (!booth)
        return res.status(404).json({ message: "booth not found" });

      const exhibition = await ExhibitionsModel.findById(booth.exhibition_id);
      const user = await UserModel.findById(booth.owner_id);
      const organization = user?.organization_id
        ? await OrganizationModel.findById(user.organization_id)
        : null;

      const videos = await VideoWallsModel.find({
        ref: "boughten_booths",
        ref_id: booth_id,
      });

      const files = await FileModel.find({ ref_id: booth_id });

      const categorized = {
        posters: [],
        monitors: [],
        stands: [],
        agents: [],
      };

      for (const f of files) {
        if (f.ref && categorized[f.ref]) {
          categorized[f.ref].push(f);
        }
      }

      return res.status(200).json({
        booth,
        exhibition,
        user,
        organization,
        videos,
        files: categorized,
      });
    } catch (err) {
      console.error("Full info error:", err);
      return res.status(500).json({ message: "Internal server error" });
    }
  }
);

router.post(
  "/boughten_booths/all",
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  async (req, res) => {
    try {
      const { owner_id } = req.body;

      if (!owner_id) {
        return res.status(400).json({
          data: null,
          message: "owner_id مشکل",
        });
      }

      const booths = await BoughtenBoothsModel.find({
        is_removed: { $ne: true },
        owner_id: owner_id,
      }).lean();

      const basePath = path.join(process.cwd(), 'uploads');
      const fileTypes = ['posters', 'video_walls', 'monitors', 'stands'];

      for (const booth of booths) {
        const boothPath = path.join(
          basePath,
          'exhibitions',
          booth.exhibition_id.toString(),
          'boughten_booths',
          booth._id.toString()
        );

        const filesByType = {};

        for (const type of fileTypes) {
          try {
            const dirPath = path.join(boothPath, type);
            const files = await fs.readdir(dirPath);
            filesByType[type] = files.map(f =>
              `http://192.168.10.50:6015/exhibitions/${booth.exhibition_id}/boughten_booths/${booth._id}/${type}/${f}`
            );
          } catch {
            filesByType[type] = [];
          }
        }

        // ✅ افزودن لوگو
        try {
          const logoDir = path.join(boothPath, "logo");
          const logoFiles = await fs.readdir(logoDir);
          if (logoFiles.length > 0) {
            filesByType["logo"] =
              `http://192.168.10.50:6015/exhibitions/${booth.exhibition_id}/boughten_booths/${booth._id}/logo/${logoFiles[0]}`;
          } else {
            filesByType["logo"] = null;
          }
        } catch {
          filesByType["logo"] = null;
        }

        // 🟣 agents/raw_files & processed_files
        const agentFiles = {};
        try {
          for (const sub of ['raw_files', 'processed_files']) {
            const agentDir = path.join(boothPath, 'agents', sub);
            const agentList = await fs.readdir(agentDir);
            agentFiles[sub] = agentList.map(f =>
              `http://192.168.10.50:6015/exhibitions/${booth.exhibition_id}/boughten_booths/${booth._id}/agents/${sub}/${f}`
            );
          }
        } catch {
          agentFiles.raw_files = agentFiles.raw_files || [];
          agentFiles.processed_files = agentFiles.processed_files || [];
        }

        booth.files = {
          ...filesByType,
          agents: agentFiles
        };
      }

      return res.status(200).json({
        data: booths,
        message: "ارسال شد",
      });
    } catch (error) {
      console.error("❌ خطا:", error);
      return res.status(500).json({
        data: null,
        message: "خطای داخلی سرور",
      });
    }
  }
);



export default router;
